package easik.sketch.util.Export.Constraints;

import java.util.LinkedList;

import easik.sketch.util.Export.ExportConstants;
import easik.sketch.util.Export.Components.ExportEdge;
import easik.sketch.util.Export.Components.ExportPath;


/**
 * A class to create a product constraint object, including all necessary information gleaned 
 * from the XML file.  Should not be used for any other purpose than database exportation, as it 
 * contains no graphical information.
 * 
 * @author Vera Ranieri 2006
 * @author Kevin Green 2006
 * @since 2006-05-22 Vera Ranieri
 * @version 2006-08-28 Kevin Green
 *
 */
public class ExportProductConstraint extends ExportConstraint {


	/**
	 * The name of the product delete constraint
	 */
	private final String DEL_PROC_NAME = "prodDel";
	/**
	 * The name of the product insert constraint
	 */
	private final String INS_PROC_NAME = "prodIns";

	
	/**
	 * Constructor calling the constructor of the ExportConstraint superclass.
	 * @param id The unique identifier for this constraint
	 */
	public ExportProductConstraint(int id) {
		super();
		_id = id;
	}

	/**
	 * Sets the base table of this product constraint.  This table is the domain of all paths
	 * included in this constraint.
	 */
	public void setBaseTable(){
		_baseTable = ((ExportPath)_paths.get(0)).getDomain();
	}
	
	/**
	 * Sets the constraint strings for this constraint 
	 * 
	 * @since 2006-06-22 Vera Ranieri
	 */
	public void setConstraintStrings(){
		setDeleteProc();
		setInsertProcs();
	}
	
	/**
	 * Sets the delete constraint string for this constraint.  
	 *
	 * @since 2006-06-22 Vera Ranieri
	 */
	private void setDeleteProc(){
		
		
		String deleteString = _db_name + "_" + DEL_PROC_NAME + _id ;
		_b_d_tables.put(_baseTable, deleteString);
		
		deleteString =ExportConstants.CREATE_PROC + deleteString+ ExportConstants.PROC_PARAM +
						ExportConstants.BEGIN;
		
		int size = _paths.size();
		for(int i =0 ; i< size; i++){
			deleteString += "DELETE FROM " + ((ExportPath)_paths.get(i)).getCoDomain() + " USING " + ExportPath.getAllTablesString((ExportPath)_paths.get(i)) 
						+"WHERE \n"
						+ ExportPath.getDelConditions((ExportPath)_paths.get(i)) + "; \n";
			
		}
		deleteString += ExportConstants.END;
		
		_procedureStrings.add(deleteString);
	}
	
	/**
	 * Sets the insert constraint strings for this constraint.
	 * 
	 * @since 2006-06-22 Vera Ranieri
	 */
	private void setInsertProcs(){
		
		
		String insertString, table, procName;
		ExportPath path;
		int size = _paths.size();
		for(int i = 0; i< size; i++){
			path = ((ExportPath)_paths.get(i));
			table = path.getCoDomain();
			procName = _db_name + "_" + INS_PROC_NAME + table+_id;
			_a_i_tables.put(table, procName);
			
			insertString = ExportConstants.CREATE_PROC + procName + ExportConstants.PROC_PARAM;
			insertString += ExportConstants.BEGIN + "DECLARE var INT; \n";
			
			insertString += getFirstIfStatement(path);

			insertString += getSecondIfStatement(path);
			
			insertString += getPathInsertStrings(path);

			insertString += getInsertCall(path);
			
			insertString += "END IF; \n";
			
			insertString += ExportConstants.END;

			_procedureStrings.add(insertString);
		}
		
	}
	/**
	 * Gets the first <b>IF</b> statement.  This statement determines whether the tables which form the domains of 
	 * the other paths involved in the product constraint contain items.  The statement is true if all tables have
	 * at least one element.
	 * @param p
	 * @return The string containing the first IF statement.
	 */
	private String getFirstIfStatement(ExportPath p){
		String first = "IF (";
		
		int size = _paths.size();
		ExportPath path;
		for(int i = 0; i < size; i ++){
			path = (ExportPath)_paths.get(i);
			if(!path.equals(p)){
				first += "( (SELECT COUNT(*) FROM " + path.getCoDomain() +") != 0) AND ";
			}
		}
		//remove 'AND ' from end.
		first = first.substring(0, first.length()-4);
		first += ") \n THEN \n";
		return first;
	}
	/**
	 * Gets the strings inserting an element for every table along the path up until the base table.  Also sets the variable 
	 * <b>var</b> to the last insert id.
	 * @param p The path upon which we need to insert
	 * @return The string of all the insert calls
	 * @since 2006-06-28 Vera Ranieri
	 */
	private String getPathInsertStrings(ExportPath p){
		
		//TODO: needs to be functional if table has many foreign keys.
		String insert = "";
		
		String target, source;
		ExportEdge e;
		LinkedList edges = p.getEdges();
		int size = edges.size();
		if( size != 1){
			e = (ExportEdge)edges.get(size-1);
			insert += "INSERT INTO " + e.getSource() +" (" + e.getTarget() + ExportConstants.ID +") " + "VALUES (id); \n";
			//For all except base table and second to last table.
			for(int i = size-2; i>0; i--){
				e = (ExportEdge)edges.get(i);
				target = e.getTarget();
				source = e.getSource();
				
				insert +="INSERT INTO " + source +" ("+ target + ExportConstants.ID+ ") " + "VALUES (LAST_INSERT_ID()); \n";
				 
			}

			insert += "SET var = LAST_INSERT_ID(); \n";
		}
		else
			insert += "SET var = id; \n";
		return insert;
	}
	/**
	 * Gets the second <b>IF</b> statement.  This statement checks if any elements need to be added along a 
	 * secondary path, before elements are added into the base table.
	 * @param p
	 * @return The string of the second IF statement.  If this statement is unneeded, returns the empty string
	 */
	private String getSecondIfStatement(ExportPath p){
		int numPaths = _paths.size();
		String second = "IF( (SELECT COUNT(*) FROM " + p.getCoDomain() +") = 1 )\n THEN \n";
		
		int size;
		boolean needed = false;
		LinkedList<ExportEdge> edges;
		ExportEdge e;
		ExportPath path;
		String target, source;
		for(int j = 0; j < numPaths; j++){
			path = _paths.get(j);
			if(!path.equals(p)){
				edges = path.getEdges();
				size = edges.size();
				for(int i = size-1; i>0; i--){
					e = edges.get(i);
					target = e.getTarget();
					source = e.getSource();
					
					second += "INSERT INTO " + source +" ("+ target + ExportConstants.ID+ ") " + "SELECT "+
							target +"." + target+ ExportConstants.ID + "FROM "+ target +"; \n";
					needed = true;
					 
				}
			}
		}
		
		second += "END IF; \n";
		if (needed)
			return second;
		else
			return "";
	}
	/**
	 * Gets the actual <b>INSERT SELECT</b> string for the constraint.
	 * @param p The path for which this constraint applies
	 * @return The string of the statement inserting into the table.
	 * @since 2006-06-28 Vera Ranieri
	 */
	private String getInsertCall(ExportPath p){
		String insert = "INSERT INTO " + p.getDomain() ;
		
		
		
		String target;
		String values ="";
		String columns = "( ";
		String from = "";
		int size = _paths.size();
		
		for(int i = 0; i< size; i++){
			target =((ExportEdge)(((ExportPath)_paths.get(i)).getEdges()).getFirst()).getTarget();
			columns += target + ExportConstants.ID + ", ";
			values += target +"." +target + ExportConstants.ID + ", ";
			from += target +", ";
		}

		columns = columns.substring(0, columns.length()-2) + ") \n";
		values = values.substring(0, values.length()-2);
		from = from.substring(0, from.length()-2) + " ";
		
		insert += columns + "SELECT " + values + "FROM " + from;
		
		String codo = ((ExportEdge)p.getEdges().getFirst()).getTarget();
		insert += "WHERE \n"+ codo +"."+ codo+ ExportConstants.ID +"=var; \n";
		return insert;
	}
	
}
